using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;

namespace BreadAndCheese {
    /// <summary>
    /// To jest gwny typ Twojej gry.
    /// </summary>
    public class BreadAndCheeseGame : Microsoft.Xna.Framework.Game {
        GraphicsDeviceManager graphics;
        SpriteBatch spriteBatch;
#region Struktury gry

        public struct BackgroundSpriteStruct {
            private Texture2D spriteTexture;
            private Rectangle spriteRectangle;

            /// <summary>
            /// aduje tekstur uywan w ramach danego sprajtu.
            /// </summary>
            /// <param name="inSpriteTexture">Uywana tekstura.</param>
            public void LoadTexture(Texture2D inSpriteTexture) {
                spriteTexture = inSpriteTexture;
            }

            /// <summary>
            /// aduje prostokt wyznaczajcy wymiary na potrzeby 
            /// operacji rysowania tego sprajtu ta.
            /// </summary>
            /// <param name="inSpriteRectangle">Prostokt, ktry ma zosta uyty.</param>
            public void SetRectangle(Rectangle inSpriteRectangle) {
                spriteRectangle = inSpriteRectangle;
            }

            /// <summary>
            /// Rysuje to na podstawie prostokta okrelajcego jej wymiary.
            /// </summary>
            /// <param name="spriteBatch">Obiekt klasy SpriteBatch, ktry ma zosta uyty
            /// przez operacj rysowania.</param>
            public void Draw(SpriteBatch spriteBatch) {
                spriteBatch.Draw(spriteTexture, spriteRectangle, Color.White);
            }
        }

        public struct TitleSpriteStruct {
            private Texture2D spriteTexture;
            private static Rectangle spriteRectangle;

            /// <summary>
            /// aduje tekstur uywan w ramach danego sprajtu.
            /// </summary>
            /// <param name="inSpriteTexture">Uywana tekstura.</param>
            public void LoadTexture(Texture2D inSpriteTexture) {
                spriteTexture = inSpriteTexture;
            }

            /// <summary>
            /// aduje prostokt wyznaczajcy wymiary na potrzeby 
            /// operacji rysowania sprajtu tytuowego.
            /// </summary>
            /// <param name="inSpriteRectangle">Prostokt, ktry ma zosta uyty.</param>
            public void SetRectangle(Rectangle inSpriteRectangle) {
                spriteRectangle = inSpriteRectangle;
            }

            /// <summary>
            /// Rysuje tytu na podstawie prostokta okrelajcego jego wymiary.
            /// </summary>
            /// <param name="spriteBatch">Obiekt klasy SpriteBatch, ktry ma zosta uyty
            /// przez operacj rysowania.</param>
            public void Draw(SpriteBatch spriteBatch) {
                spriteBatch.Draw(spriteTexture, spriteRectangle, Color.White);
            }

            /// <summary>
            /// Aktualizuje zachowanie dla tytuu. Metoda testuje stan przycisku A i uruchamia gr,
            /// jeli ten przycisk jest wcinity.
            /// </summary>
            /// <param name="game">Gra, ktrej przebieg ma by kontrolowany.</param>
            public void Update(BreadAndCheeseGame game) {
                if (game.GamePad1.Buttons.A == ButtonState.Pressed) {
                    game.StartGame();
                }
            }
        }

        public struct BatSpriteStruct {
            private Texture2D spriteTexture;
            private Rectangle spriteRectangle;
            private float x;
            private float y;
            private float xSpeed;
            private float ySpeed;

            private float minDisplayX;
            private float maxDisplayX;
            private float minDisplayY;
            private float maxDisplayY;

            /// <summary>
            /// aduje tekstur do sprajtu paki.
            /// </summary>
            /// <param name="inSpriteTexture">tekstura do zaadowania</param>
            public void LoadTexture(Texture2D inSpriteTexture) {
                spriteTexture = inSpriteTexture;
            }

            /// <summary>
            /// Metoda rozpoczyna gr. Wyznacza rozmiary wszystkich obiektw na 
            /// ekranie i umieszcza na pocztkowych pozycjach.
            /// </summary>
            /// <param name="widthFactor">Rozmiar obiektu w grze wyraony jako uamek szerokoci
            /// ekranu.</param>
            /// <param name="ticksToCrossScreen">Szybko poruszania si obiektu wyraona w liczbie taktw (1/60 sekundy) 
            /// potrzebnych do pokonania caego ekranu.</param>
            /// <param name="inMinDisplayX">minimalna warto X</param>
            /// <param name="inMaxDisplayX">maksymalna warto X</param>
            /// <param name="inMinDisplayY">minimalna warto Y</param>
            /// <param name="inMaxDisplayY">maksymalna warto Y</param>
            /// <param name="initialX">pocztkowa pozycja paki (X)</param>
            /// <param name="initialY">pocztkowa pozycja paki (Y)</param>

            public void StartGame(
                    float widthFactor,
                    float ticksToCrossScreen,
                    float inMinDisplayX,
                    float inMaxDisplayX,
                    float inMinDisplayY,
                    float inMaxDisplayY,
                    float initialX,
                    float initialY) {
                minDisplayX = inMinDisplayX;
                minDisplayY = inMinDisplayY;
                maxDisplayX = inMaxDisplayX;
                maxDisplayY = inMaxDisplayY;

                float displayWidth = maxDisplayX - minDisplayX;

                spriteRectangle.Width = (int) ((displayWidth * widthFactor) + 0.5f);
                float aspectRatio =
                        (float) spriteTexture.Width / spriteTexture.Height;
                spriteRectangle.Height =
                        (int) ((spriteRectangle.Width / aspectRatio) + 0.5f);
                x = initialX;
                y = initialY;
                xSpeed = displayWidth / ticksToCrossScreen;
                ySpeed = xSpeed;
            }

            /// <summary>
            /// Rysuje pak na ekranie.
            /// </summary>
            /// <param name="spriteBatch">Obiekt klasy SpriteBatch uywany do rysowania na ekranie</param>
            public void Draw(SpriteBatch spriteBatch) {
                spriteBatch.Draw(spriteTexture, spriteRectangle, Color.White);
            }

            /// <summary>
            /// Aktualizuje pooenie paki zalenie od stanu pada.
            /// </summary>
            public void Update(BreadAndCheeseGame game) {
                GamePadState gamePad1 = GamePad.GetState(PlayerIndex.One);

                x = x + (xSpeed * gamePad1.ThumbSticks.Left.X);
                y = y - (ySpeed * gamePad1.ThumbSticks.Left.Y);
                spriteRectangle.X = (int) x;
                spriteRectangle.Y = (int) y;
            }

            /// <summary>
            /// Sprawdza ewentualne kolizje pomidzy pak a pozostaymi obiektami.
            /// </summary>
            /// <param name="target">Prostokt reprezentujcy pooenie innego obiektu.</param>
            /// <returns>true, jeli paka koliduje z danym obiektem</returns>
            public bool CheckCollision(Rectangle target) {
                return spriteRectangle.Intersects(target);
            }
        }

        public struct BallSpriteStruct {
            private Texture2D spriteTexture;
            private Rectangle spriteRectangle;
            private float x;
            private float y;
            private float xSpeed;
            private float ySpeed;

            private float minDisplayX;
            private float maxDisplayX;
            private float minDisplayY;
            private float maxDisplayY;

            /// <summary>
            /// aduje tekstur do sprajtu piki.
            /// </summary>
            /// <param name="inSpriteTexture">tekstura do zaadowania</param>
            public void LoadTexture(Texture2D inSpriteTexture) {
                spriteTexture = inSpriteTexture;
            }

            /// <summary>
            /// Metoda rozpoczyna gr. Wyznacza rozmiary wszystkich obiektw na 
            /// ekranie i umieszcza na pocztkowych pozycjach.
            /// </summary>
            /// <param name="widthFactor">Rozmiar obiektu w grze wyraony jako uamek szerokoci
            /// ekranu.</param>
            /// <param name="ticksToCrossScreen">Szybko poruszania si obiektu wyraona w liczbie taktw (1/60 sekundy) 
            /// potrzebnych do pokonania caego ekranu.</param>
            /// <param name="inMinDisplayX">minimalna warto X</param>
            /// <param name="inMaxDisplayX">maksymalna warto X</param>
            /// <param name="inMinDisplayY">minimalna warto Y</param>
            /// <param name="inMaxDisplayY">maksymalna warto Y</param>
            /// <param name="initialX">pocztkowa pozycja piki (X)</param>
            /// <param name="initialY">pocztkowa pozycja piki (Y)</param>

            public void StartGame(
                    float widthFactor,
                    float ticksToCrossScreen,
                    float inMinDisplayX,
                    float inMaxDisplayX,
                    float inMinDisplayY,
                    float inMaxDisplayY,
                    float initialX,
                    float initialY) {
                minDisplayX = inMinDisplayX;
                minDisplayY = inMinDisplayY;
                maxDisplayX = inMaxDisplayX;
                maxDisplayY = inMaxDisplayY;

                float displayWidth = maxDisplayX - minDisplayX;

                spriteRectangle.Width = (int) ((displayWidth * widthFactor) + 0.5f);
                float aspectRatio =
                        (float) spriteTexture.Width / spriteTexture.Height;
                spriteRectangle.Height =
                        (int) ((spriteRectangle.Width / aspectRatio) + 0.5f);
                x = initialX;
                y = initialY;
                xSpeed = displayWidth / ticksToCrossScreen;
                ySpeed = xSpeed;
            }

            /// <summary>
            /// Rysuje pik na ekranie.
            /// </summary>
            /// <param name="spriteBatch">Obiekt klasy SpriteBatch uywany do rysowania na ekranie</param>
            public void Draw(SpriteBatch spriteBatch) {
                spriteBatch.Draw(spriteTexture, spriteRectangle, Color.White);
            }

            /// <summary>
            /// Aktualizuje pooenie piki. Obsuguje kolizje z pak.
            /// </summary>
            /// <param name="game">Gra, w ktrej wystpuje ta pika</param>
            public void Update(BreadAndCheeseGame game) {
                x = x + xSpeed;
                y = y + ySpeed;

                // Ustawia nowe pooenie prostokta sprajtu:
                spriteRectangle.X = (int) (x + 0.5f);
                spriteRectangle.Y = (int) (y + 0.5f);

                // Sprawdza, czy paka trafia w pik:
                if (game.BreadBat.CheckCollision(spriteRectangle)) {
                    // Paka trafia w pik.
                    ySpeed = ySpeed * -1;
                }

                if (x + spriteRectangle.Width >= maxDisplayX) {
                    // Pika zostaa trafiona z prawej strony.
                    xSpeed = Math.Abs(xSpeed) * -1;
                }

                if (x <= minDisplayX) {
                    // Pika zostaa trafiona z lewej strony.
                    xSpeed = Math.Abs(xSpeed);
                }

                // Sprawdza, czy pika trafia w doln krawd ekranu.
                if (y + spriteRectangle.Height >= maxDisplayY) {
                    // Pika zderzya si z doln krawdzi. Powoduje utrat ycia.
                    ySpeed = Math.Abs(ySpeed) * -1;
                    game.LoseLife();
                }

                if (y <= minDisplayY) {
                    // Pika uderzya w grn cz ekranu.
                    ySpeed = Math.Abs(ySpeed);
                }

                if (game.TomatoTargets.CheckCollision(spriteRectangle)) {
                    game.UpdateScore(20);
                    ySpeed = ySpeed * -1;
                }

            }
        }

        public struct TargetRowStruct {
            private Texture2D targetTexture;
            private int numberOfTargets;
            private float targetWidth;
            private float targetHeight;
            private float targetStepFactor;
            private float targetHeightLimit;
            private Rectangle[] targets;
            private bool[] targetVisibility;

            private float minDisplayX;
            private float maxDisplayX;
            private float minDisplayY;
            private float maxDisplayY;
            private float displayHeight;
            private float displayWidth;

            public void LoadTexture(Texture2D inTargetTexture) {
                targetTexture = inTargetTexture;
            }

            public void StartGame(
                    int inNumberOfTargets,
                    float inTargetStepFactor,
                    float inMinDisplayX,
                    float inMaxDisplayX,
                    float inMinDisplayY,
                    float inMaxDisplayY) {
                numberOfTargets = inNumberOfTargets;
                targetStepFactor = inTargetStepFactor;

                minDisplayX = inMinDisplayX;
                minDisplayY = inMinDisplayY;
                maxDisplayX = inMaxDisplayX;
                maxDisplayY = inMaxDisplayY;

                displayWidth = maxDisplayX - minDisplayX;
                displayHeight = maxDisplayY - minDisplayY;

                targetWidth = displayWidth / numberOfTargets;

                float aspectRatio = (float) targetTexture.Width / targetTexture.Height;

                targetHeight = targetWidth / aspectRatio;

                targetHeightLimit = minDisplayY + ((maxDisplayY - minDisplayY) / 2);

                targets = new Rectangle[numberOfTargets];
                targetVisibility = new bool[numberOfTargets];

                for (int i = 0; i < numberOfTargets; i++) {
                    targets[i].Width = (int) targetWidth;
                    targets[i].Height = (int) targetHeight;

                    targets[i].Y = (int) targetHeight;
                    targets[i].X = (int) (minDisplayX + (i * targetWidth) + 0.5f);

                    targetVisibility[i] = true;
                }
            }

            void resetTargetDisplay() {
                targetHeight = targetHeight + (displayHeight * targetStepFactor);

                if (targetHeight > targetHeightLimit) {
                    targetHeight = minDisplayY;
                }

                for (int i = 0; i < numberOfTargets; i++) {
                    targets[i].Y = (int) targetHeight;
                    targetVisibility[i] = true;
                }
            }

            public void Update(BreadAndCheeseGame game) {
                for (int i = 0; i < numberOfTargets; i++) {
                    if (targetVisibility[i]) {
                        // Zwraca sterowanie w razie znalezienia widocznego celu.
                        return;
                    }
                }

                // Jeli sterowanie dotaro w to miejsce, nie ma widocznych pomidorw.

                // Obnia miejsce rysowania celu na ekranie.
                targetHeight = targetHeight + (displayHeight * targetStepFactor);

                // Sprawdza, czy osignito doln granic rysowania celw.
                if (targetHeight > targetHeightLimit) {
                    targetHeight = minDisplayY;
                }

                // Przywraca ustawienia wszystkich celw.
                for (int i = 0; i < numberOfTargets; i++) {
                    targets[i].Y = (int) targetHeight;
                    targetVisibility[i] = true;
                }
            }

            public void Draw(SpriteBatch spriteBatch) {
                for (int i = 0; i < numberOfTargets; i++) {
                    if (targetVisibility[i]) {
                        spriteBatch.Draw(targetTexture, targets[i], Color.White);
                    }
                }
            }

            public bool CheckCollision(Rectangle target) {
                for (int i = 0; i < numberOfTargets; i++) {
                    if (targetVisibility[i]) {
                        // Uzyskuje cel uczestniczcy w kolizji.
                        if (targets[i].Intersects(target)) {
                            // Niszczy pomidor.
                            targetVisibility[i] = false;
                            // Zwraca warto reprezentujc wystpienie kolizji.
                            return true;
                        }
                    }
                }
                return false;
            }
        }

#endregion

        // wiat gry
        public GamePadState GamePad1;

        public BatSpriteStruct BreadBat;
        public BallSpriteStruct CheeseBall;
        public TargetRowStruct TomatoTargets;
        public TitleSpriteStruct Title;
        public BackgroundSpriteStruct Background;

#region Wartoci reprezentujce wymiary ekranu

        int displayWidth;
        int displayHeight;
        float overScanPercentage = 10.0f;
        float minDisplayX;
        float maxDisplayX;
        float minDisplayY;
        float maxDisplayY;

        private void setScreenSizes() {
            displayWidth = graphics.GraphicsDevice.Viewport.Width;
            displayHeight = graphics.GraphicsDevice.Viewport.Height;
            float xOverscanMargin = getPercentage(overScanPercentage, displayWidth) / 2.0f;
            float yOverscanMargin = getPercentage(overScanPercentage, displayHeight) / 2.0f;

            minDisplayX = xOverscanMargin;
            minDisplayY = yOverscanMargin;

            maxDisplayX = displayWidth - xOverscanMargin;
            maxDisplayY = displayHeight - yOverscanMargin;
        }


#endregion

#region Metody pomocnicze

        /// <summary>
        /// Oblicza wartoci procentowe
        /// </summary>
        /// <param name="percentage">procent, ktry ma zosta wyznaczony</param>
        /// <param name="inputValue">warto do przeliczenia</param>
        /// <returns>warto wynikowa</returns>

        float getPercentage(float percentage, float inputValue) {
            return (inputValue * percentage) / 100;
        }

#endregion

#region Rysowanie tekstu

        SpriteFont font;

        private void loadFont() {
            font = Content.Load<SpriteFont > ("SpriteFont1");
        }

        /// <summary>
        /// Rysuje tekst na ekranie
        /// </summary>
        /// <param name="text">tekst do wywietlenia</param>
        /// <param name="textColor">kolor tekstu</param>
        /// <param name="x">lewa krawd tekstu</param>
        /// <param name="y">grna krawd tekstu</param>

        void drawText(string text, Color textColor, float x, float y) {
            int layer;
            Vector2 textVector = new Vector2(x, y);

            // Rysuje cie
            Color backColor = new Color(0, 0, 0, 10);
            for (layer = 0; layer < 10; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X++;
                textVector.Y++;
            }

            // Rysuje podstaw znakw
            backColor = new Color(190, 190, 190);
            for (layer = 0; layer < 5; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X++;
                textVector.Y++;
            }

            // Rysuje przedni cz znakw
            spriteBatch.DrawString(font, text, textVector, textColor);
        }

#endregion

#region Rekordowy wynik

        void updateScore() {
        }

        void drawScore() {
            drawText("Punkty: " + score + " ycia: " + lives, Color.Blue, minDisplayX, maxDisplayY - 50);
        }

        void drawHighScore() {
            drawText("Najlepszy wynik: " + highScore + " Nacinij A, aby rozpocz", Color.Blue, minDisplayX, minDisplayY);
        }

#endregion

#region Zarzdzanie stanem gry
        int score;
        int lives;
        int highScore;

        enum GameState {
            titleScreen,
            playingGame
        }

        GameState state = GameState.titleScreen;

        public void StartGame() {
            score = 0;
            lives = 3;

            CheeseBall.StartGame(
                    0.07f, // szeroko sera wynosi 0,07 szerokoci ekranu
                    200, // ser potrzebuje 200 taktw na pokonanie caego ekranu
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    displayWidth / 4, // jedna czwarta szerokoci ekranu
                    displayHeight / 4); // jedna czwarta wysokoci ekranu od grnej krawdzi

            BreadBat.StartGame(
                    0.166f, // szeroko chleba wynosi 0,166 szerokoci ekranu
                    150, // pokonanie ekranu zajmuje 150 taktw
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    displayWidth / 2, // na pocztku znajduje si na rodku ekranu (w poziomie)
                    displayHeight / 2); // na pocztku znajduje si na rodku ekranu (w pionie)

            TomatoTargets.StartGame(
                    20,
                    0.1f,
                    minDisplayX,
                    maxDisplayX,
                    minDisplayY,
                    maxDisplayY);

            state = GameState.playingGame;
        }

        void gameOver() {
            if (score > highScore) {
                highScore = score;
            }
            state = GameState.titleScreen;
        }

        public void LoseLife() {
            lives--;
            if (lives == 0) {
                gameOver();
            }
        }

        public void UpdateScore(int update) {
            score = score + update;
        }

#endregion


        public BreadAndCheeseGame() {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";
        }

        /// <summary>
        /// Umoliwia ewentualn inicjalizacj przed uruchomieniem waciwej gry.
        /// W tym miejscu mona odnale wszystkie wymagane zasoby i zaadowa treci
        /// related content. Wywoanie metody base.Initialize spowoduje przeszukanie wszystkich komponentw
        /// i ich inicjalizacj.
        /// </summary>
        protected override void Initialize() {
            setScreenSizes();

            // wypenia widoczny obszar tekstur tytuow
            Title.SetRectangle(
                    new Rectangle(
                    (int) minDisplayX, (int) minDisplayY,
                    (int) (maxDisplayX - minDisplayX),
                    (int) (maxDisplayY - minDisplayY)
                    ));


            // wypenia widoczny obszar tekstur ta
            Background.SetRectangle(
                    new Rectangle(
                    (int) minDisplayX, (int) minDisplayY,
                    (int) (maxDisplayX - minDisplayX),
                    (int) (maxDisplayY - minDisplayY)
                    ));

            base.Initialize();
        }

        /// <summary>
        /// Metoda LoadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu adujcego tre.
        /// </summary>
        protected override void LoadContent() {
            // Tworzy nowy obiekt klasy SpriteBatch, ktrego mona uywa do rysowania tekstur.
            spriteBatch = new SpriteBatch(GraphicsDevice);

            BreadBat.LoadTexture(Content.Load<Texture2D > ("Images/Bread"));
            CheeseBall.LoadTexture(Content.Load<Texture2D > ("Images/Cheese"));
            TomatoTargets.LoadTexture(Content.Load<Texture2D > ("Images/Tomato"));
            Title.LoadTexture(Content.Load<Texture2D > ("Images/Title"));
            Background.LoadTexture(Content.Load<Texture2D > ("Images/Background"));
            loadFont();
        }

        /// <summary>
        /// Metoda UnloadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu usuwajcego ca tre z pamici.
        /// </summary>
        protected override void UnloadContent() {
            // TODO: Naley usun ca tre, ktr nie zarzdza ContentManager.
        }

        /// <summary>
        /// Umoliwia grze wykonywanie logiki zwizanej z aktualizacj wiata gry,
        /// sprawdzaniem kolizji, pobieraniem danych wejciowych czy odtwarzaniem dwikw.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Update(GameTime gameTime) {
            GamePad1 = GamePad.GetState(PlayerIndex.One);

            if (GamePad1.Buttons.Back == ButtonState.Pressed)
                this.Exit();

            switch (state) {
                case GameState.titleScreen:
                    Title.Update(this);
                    break;
                case GameState.playingGame:
                    BreadBat.Update(this);
                    CheeseBall.Update(this);
                    TomatoTargets.Update(this);
                    updateScore();
                    break;
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// Ta metoda jest wywoywana w momencie, w ktrym gra musi narysowa swj wiat.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Draw(GameTime gameTime) {
            GraphicsDevice.Clear(Color.CornflowerBlue);

            spriteBatch.Begin();

            switch (state) {
                case GameState.titleScreen:
                    Title.Draw(spriteBatch);
                    drawHighScore();
                    break;
                case GameState.playingGame:
                    Background.Draw(spriteBatch);
                    BreadBat.Draw(spriteBatch);
                    CheeseBall.Draw(spriteBatch);
                    TomatoTargets.Draw(spriteBatch);
                    drawScore();
                    break;
            }

            spriteBatch.End();

            base.Draw(gameTime);
        }
    }
}
